#ifndef UNICAN_ARD_H_
#define UNICAN_ARD_H_
#include "atmega2560/unican.h"
#include "CAN.h"

#define DEFAULT_SELF_ID 0x1A
#define DEFAULT_MASTER_ID 0x05
#define SW_VERSION {0, 1, 0, "\0"};

#define ACK_MSGID     0x0118
#define NACK_MSGID    0x0119

/**
* @brief Class for Arduino implemented Unican Device 
*/
class UnicanArdDevice
{
public:
    
    /**
    * @brief UnicanArdDevice default constructor
    */
    UnicanArdDevice();  
    
    /**
    * @brief UnicanArdDevice custom constructor
    * @param self_id self unican identifier
    * @return error code (0 for succesfull execution) 
    */
    UnicanArdDevice(uint8_t self_id);   
    
    /**
    * @brief UnicanArdDevice custom constructor
    * @param self_id self unican identifier
    * @param master_id self unican identifier
    * @return error code (0 for succesfull execution) 
    */
    UnicanArdDevice(uint8_t self_id, uint8_t master_id);   
    
    /**
    * @brief UnicanArdDevice initialisation
    * @return error code (0 for succesfull execution) 
    */
    uint8_t Init();
    
    /**
    * @brief UnicanArdDevice change unican identifier
    * @param self_id new self unican identifier
    * @return error code (0 for succesfull execution) 
    */
    uint8_t ChangeSelfId(uint8_t self_id);
    
    /**
    * @brief Change master's unican identifier
    * @param master_id new master unican identifier
    * @return error code (0 for succesfull execution) 
    */
    uint8_t ChangeMasterId(uint8_t master_id);
    
        
    /**
    * @brief UnicanArdDevice deinitialisation
    * @return error code (0 for succesfull execution) 
    */
    uint8_t DeInit();
    
        
    /**
    * @brief Receive income message if any (put this inside short period loop)
    * @return error code (0 for succesfull execution) 
    */
    uint8_t ReceiveTask();
    
    /**
    * @brief Send unican message
    * @param receiver_id unican identifier of destination (master) device
    * @param msg_id message identification code
    * @param data pointer to message data
    * @param length length of data structure in bytes
    * @return error code (0 for succesfull execution) 
    */
    uint8_t SendMessage(uint8_t receiver_id, uint16_t msg_id, uint8_t* data, uint8_t length);
    
    /**
    * @brief Send unican message to master
    * @param msg_id message identification code
    * @param data pointer to message data
    * @param length length of data structure in bytes
    * @return error code (0 for succesfull execution) 
    */
    uint8_t SendMessageToMaster(uint16_t msg_id, uint8_t* data, uint8_t length);
    
    /**
    * @brief Save receive unican message (function is called automatically)
    * @param unican_message pointer received message
    * @return error code (0 for succesfull execution) 
    */
    uint8_t SaveMessage(unican_message* msg); 
    
    /**
    * @brief Receive last unican message
    * @param sender_id pointer to unican identifier of source device
    * @param msg_id pointer to message identification code
    * @param data pointer to message data
    * @param length pointer to length of data structure in bytes
    * @return error code (0 for succesfull execution, 1 for empty receive buffer) 
    */
    uint8_t ReadMessage(uint8_t* sender_id, uint16_t* msg_id, uint8_t* data, uint8_t* length);
    
    
    
    /**
    * @brief Return acknoledge message
    * @param receiver_id unican identifier of destination (master) device
    * @return error code (0 for succesfull execution) 
    */
    uint8_t SendAck(uint8_t receiver_id); 
    
    
    
    /**
    * @brief Return not acknoledge message
    * @param receiver_id unican identifier of destination (master) device
    * @return error code (0 for succesfull execution) 
    */
    uint8_t SendNack(uint8_t receiver_id); 
    
private:
    uint8_t _self_id, _master_id;
    MCP2515Class* _CAN;
    unican_message _received_msg;
    uint8_t _received_msg_data_buffer[255];
    
    uint8_t SetFilter();
};

extern UnicanArdDevice UCanArd;

#endif /* UNICAN_ARD_H_ */
